%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Function name: read_picarro_mpv
%
% Scope:    import multiposition valve (MPV) data from external PICARRO
%           data file
%
% Input:    file name MPV data file (string),
%           directory MPV data file (string),
%           index plot requested (double)
%
% Output:   updated data struture with replaced MPV position data,
%           overview figure (if requested)   
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%
function data = read_picarro_mpv(data, filename_mpv,pathname_mpv, plot_requested)

tic;fprintf('Running function read_picarro_mpv...\n');

datatable = readtable([pathname_mpv filename_mpv]);

data_mpv.all(:,1) = datatable.('FRAC_HRS_SINCE_JAN1');     %absolute time in h
data_mpv.all(:,2) = data_mpv.all(:,1)-data.all(1,1);         %relative time in h  
data_mpv.all(:,3) = datatable.('MPVPosition'); 

%% interpolate mpv data to main instrument time grid

for i=1:height(data.all)
        [~,minidx] = min(abs(data_mpv.all(:,2)-data.all(i,2)));    
    data_mpv.intpl(i,1)=data_mpv.all(minidx,2);
    data_mpv.intpl(i,2)=data_mpv.all(minidx,3);
    data_mpv.check_idx(i,1)=minidx;
end

%% replace mpv data into main instrument

data.all(:,12)=data_mpv.intpl(:,2);

%% find interval boundaries (i.e. MPV position changes)

data.boundaries_idx = find(ischange(data.all(:,12))==1);
data.boundaries_idx = transpose([1 transpose([data.boundaries_idx]) height(data.all)]);

%% calculate mean time resolution

delta_t = diff(data.all(:,2))*60*60;    %s  
data.time_res=mean(delta_t);       

%% remove MPV "transition" switches (i.e. MPV position changes closer in time than 20 s)

discard_idx=round(20/data.time_res);

 check_boundaries=[];
 for i=1:height(data.boundaries_idx)-1
    if data.boundaries_idx(i+1,1)-data.boundaries_idx(i,1)<discard_idx
        check_boundaries=[check_boundaries find(data.boundaries_idx==data.boundaries_idx(i+1,1))];
    end
 end
 
 data.boundaries_idx(check_boundaries,1)=NaN;
 data.boundaries_idx(end,1)=height(data.all);
 data.boundaries_idx=rmmissing(data.boundaries_idx);

 %% create "flag" vector

for k=1:height(data.boundaries_idx)-1
    num=0;sum=0;
    for i=data.boundaries_idx(k,1):data.boundaries_idx(k+1,1)
        num=num+1;
        sum=sum+data.all(i,12);
    end
    data.flag(k,1)=round(sum/num);
end
%% plot overview

plotvars=[3 7 5 8 6 11];
if plot_requested
    figure;
    for i=1:6
        subplot(3,2,i);
        plot(data.all(:,2),data.all(:,plotvars(1,i)),'LineWidth',1.5);
        hold on;
        for j=2:height(data.boundaries_idx)-1
            xline(data.all(data.boundaries_idx(j,1),2),'--k');
        end
        %xline(data.all(data.boundaries_idx(2:height(data.boundaries_idx)-1,1),2),'--k');
        grid on;
        box on;
        xlim([0 data.all(end,2)]);
        ylabel({sprintf('%s',data.varlabels{1,plotvars(1,i)})},'fontsize',12);
        if i==1
        title('Overview PICARRO G5131-{\iti}','fontsize',12); end    
        if i==5 || i==6
        xlabel({sprintf('%s',data.varlabels{1,2})},'fontsize',12); end
    end
    
    x0 = 10;
    y0 = 50;
    ww = 900;
    hh = 700;
    set(gcf,'units','points','position',[x0,y0,ww,hh]);
    set(gcf,'Units','Inches');
    pos = get(gcf,'Position');
    set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3),pos(4)]);
end
%print(gcf,'C:\Users\brsi\Desktop\overview_N2O','-dpng', '-r800');

%%
time_elapsed=toc; fprintf('read_picarro_mpv completed (execution time: %1.2f s)\n',time_elapsed); 

end